/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   LED Resource Framework API Header File
::   Copyright   :   (C)2004-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_LED.h
    \brief The BEHAVIOUR_LED provides the framework with the capability to drive a tri-colour LED resource
*/
    
#ifndef __RESOURCE_LED_H
#define __RESOURCE_LED_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <resource.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/
/*! brief Describes the full scale intensity of an LED colour. A percentage multiplied by this value and then
          divided by 100 will scale intensity for use with the \ref LEDIntensity_U type. */
#define LED_INTENSITY_FULL_SCALE          ((uint4)256)


/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid LED
           attributes in the S_LEDAttributes and S_LEDCreateAttributes data structures.

Each mask definition applies to a particular member of one of the data structures.
    \code
    ...
    S_LEDCreateAttributes CreateObj;

    // Will use DynamicObj
    CreateObj.uValidAttributesMask = USE_LED_DYNAMIC_ON_CREATE;
    // Will use the condition within the dynamic object
    CreateObj.DynamicObj.uValidAttributesMask = USE_LED_CONDITION;
    CreateObj.DynamicObj.eResourceCondition = RES_ENABLED;
    // The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */    
typedef enum
{
    USE_LED_CONDITION = 0x01,            /*!< Selects S_LEDAttributes::eCondition */
    USE_LED_DYNAMIC_ON_CREATE = 0x02,    /*!< Selects S_LEDCreateAttributes::DynamicObj */
    USE_LED_INTENSITY = 0x04,            /*!< Selects S_LEDAttributes::IntensityObj */
    USE_LED_FLASHRATE = 0x08,            /*!< Selects S_LEDAttributes::FlashRateObj */
    USE_LED_INTENSITY_SPACE = 0x10,      /*!< Selects S_LEDAttributes::SpaceIntensityObj */

/* IF THIS TYPE EXCEEDS 0x80 THEN ALTER THE TYPE OF THE U_LEDAttributesMask MEMBER ACCORDINGLY */

} E_LEDAttributesMask;

typedef uint1 U_LEDAttributesMask;

/*! Describes a LED intensity. This value, when multiplied by 100 and divided by \ref LED_INTENSITY_FULL_SCALE
    will give the intensity as a percentage. */
typedef uint2 LEDIntensity_U;

/*! \brief Describes the intensity settings of a LED. Allows for a single, bi and tri-colour control control. 

           A non-zero intensity will equate to 100% ON for LEDs that are driven by hardware that is incapable
           of intensity control. */
typedef struct
{
    LEDIntensity_U uRed;     /*!< Red intensity setting in Tri, Bi and single colour LEDs. */
    LEDIntensity_U uGreen;   /*!< Green intensity setting in Tri and Bi colour LEDs. This setting is ignored
                                  in single colour LEDs */
    LEDIntensity_U uBlue;    /*!< Blue intensity setting in Tri-colour LEDs. This setting is ignored for Bi
                                  and single colour LEDs */
} S_LEDIntensity;

/*! \brief Describes the flash rate settings a LED. Allows for programmable mark/space ratio. */
typedef struct
{
    uint2 uMarkTime;            /*!< The mark time in mS to flash ON */
    uint2 uSpaceTime;           /*!< The space time in mS to flash OFF */
} S_LEDFlashRate;

/*! \brief This data structure describes all of the LED resource's runtime configuration attributes.

    The attributes are altered through the use of SetResourceAttributesBEHAVIOUR_LED(). The data
    structure does not need to be completely filled inorder to be used. The \b uValidAttributesMask is a bit
    field member that is used to identify which attributes are valid. Each attribute is identified with a
    separate bit mask that is logic-ORed into the a mask when an attribute is to be used.
    \code
    ...
    S_LEDAttributes LEDObj;

    // Only going to set the condition member
    LEDObj.uValidAttributesMask = USE_LED_CONDITION;  \endcode */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_LED_CONDITION, \ref USE_LED_INTENSITY, \ref USE_LED_FLASHRATE and \ref USE_LED_INTENSITY_SPACE]
    that are valid for this this instance of the data structure. The valid attributes are described by the \ref E_LEDAttributesMask
    enumeration
    \code
    ...
    S_LEDAttributes LEDObj;

    // Only going to set the condition member
    LEDObj.uValidAttributesMask = USE_LED_CONDITION;  \endcode */
    uint1 uValidAttributesMask;
/*! Set to \ref RES_ENABLED to allow the other assigned attributes of the LED to go into affect. Select this attribute
    with the \ref USE_LED_CONDITION bit mask. This attribute has the value of \ref RES_DISABLED when the behaviour
    is initially created. */
    E_ResourceCond eCondition;
/*! Describes the intensity of the LED during the Mark portion of a flash or the intensity if flashing is not defined. Select
    this attribute with the \ref USE_LED_INTENSITY bit mask. This attribute is set such that the LED will appear OFF (black)
    when the behaviour is initially created. */
    S_LEDIntensity IntensityObj;
/*! Describes the flash rate of the LED. Select this attribute with the \ref USE_LED_FLASHRATE bit mask. Set the mark and
    space times both to zero to disable any flashing (the default state) and leave the LED driven based upon the values of
    S_LEDAttributes::IntensityObj. */
    S_LEDFlashRate FlashRateObj;
/*! Describes the intensity of the LED during the Space portion of a flash. Set to zero to have the LED turn OFF during the
    Space portion. Select this attribute with the \ref USE_LED_INTENSITY_SPACE bit mask. This attribute
    is set such that the LED will appear OFF (black) when the behaviour is initially created. */
    S_LEDIntensity SpaceIntensityObj;

} S_LEDAttributes;

typedef S_LEDAttributes const* S_LEDAttributesPtr;

/*! \brief This data structure describes the creation attributes for a LED resource */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_LED_DYNAMIC_ON_CREATE] that are valid for this this instance of the data structure.
    The valid attributes are described by the \ref E_LEDAttributesMask enumeration

    \code
    ...
    S_LEDCreateAttributes CreateObj;

    // Will use DynamicObj
    CreateObj.uValidAttributesMask = USE_LED_DYNAMIC_ON_CREATE;
    // Dynamic object will use the condition and intensity settings
    CreateObj.DynamicObj.uValidAttributesMask = USE_LED_CONDITION | USE_LED_INTENSITY | USE_LED_FLASHRATE;
    CreateObj.DynamicObj.eResourceCondition = RES_ENABLED;
    CreateObj.DynamicObj.IntensityObj.uRed = 100;
    CreateObj.DynamicObj.IntensityObj.uGreen = 100;
    CreateObj.DynamicObj.IntensityObj.uBlue = 100;
    CreateObj.DynamicObj.SpaceIntensityObj.uRed = 0;
    CreateObj.DynamicObj.SpaceIntensityObj.uGreen = 0;
    CreateObj.DynamicObj.SpaceIntensityObj.uBlue = 0;

    // And flash at 1Hz, 60% duty cycle
    CreateObj.DynamicObj.FlashRateObj.uMarkTime = 600;
    CreateObj.DynamicObj.FlashRateObj.uSpaceTime = 400;

    // The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */    
    uint1 uValidAttributesMask;
/*! Initial values of the runtime attributes, which can be later altered through a call to
    SetResourceAttributesBEHAVIOUR_LED(). Select with the \ref USE_LED_DYNAMIC_ON_CREATE bit mask */
    S_LEDAttributes DynamicObj;
} S_LEDCreateAttributes;

typedef S_LEDCreateAttributes const* S_LEDCreateAttributesPtr;

#pragma pack()
/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/
NativeError_S CreateResourceBEHAVIOUR_LED(E_ModuleResource, S_LEDCreateAttributes const*);
NativeError_S DestroyResourceBEHAVIOUR_LED(E_ModuleResource);
NativeError_S SetResourceAttributesBEHAVIOUR_LED(E_ModuleResource, S_LEDAttributes const*);

#endif /* __RESOURCE_LED_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
